<?php
// Run via CLI (cPanel cron recommended): php /path/to/project/cron/daily.php
date_default_timezone_set('Asia/Dhaka');

require_once __DIR__ . '/../app/core/db.php';
require_once __DIR__ . '/../app/core/helpers.php';
require_once __DIR__ . '/../app/core/billing.php';
require_once __DIR__ . '/../app/core/notifications.php';
require_once __DIR__ . '/../app/core/telegram.php';
require_once __DIR__ . '/../app/core/audit.php';

function log_line(string $s): void {
  echo '['.date('Y-m-d H:i:s').'] '.$s.PHP_EOL;
}

$today = date('Y-m-d');
$day = (int)date('j');
$periodCurrent = date('Y-m'); // invoice period (current month)
$periodNext = date('Y-m', strtotime('first day of next month')); // snapshot_for_period_ym

log_line("Cron daily start. today={$today} day={$day} periodCurrent={$periodCurrent} periodNext={$periodNext}");

// ---------- (A) 25th: snapshot device count for NEXT month ----------
if ($day === 25) {
  log_line("Running billing snapshot for period {$periodNext}...");
  $tenants = db_all("SELECT id,name,status FROM tenants WHERE status!='DISABLED' ORDER BY id ASC");
  foreach ($tenants as $t) {
    $tid = (int)$t['id'];
    if (billing_snapshot_exists($tid, $periodNext)) {
      log_line(" - Tenant {$tid}: snapshot exists, skip");
      continue;
    }
    create_billing_snapshot($tid, $periodNext);
    notify_create($tid, null, 'BILLING_SNAPSHOT', 'Billing snapshot created', "Snapshot created for {$periodNext}.", 'index.php?page=invoices');
    telegram_send($tid, "📌 Billing snapshot created\nPeriod: {$periodNext}\nTenant: {$t['name']} (#{$tid})");
    log_line(" - Tenant {$tid}: snapshot created");
    audit_log($tid, null, 'BILLING_SNAPSHOT_CREATED', 'billing_snapshot', null, ['period'=>$periodNext]);
  }
}

// ---------- (B) 1st: generate invoices for CURRENT month using snapshot ----------
if ($day === 1) {
  log_line("Generating invoices for period {$periodCurrent}...");
  $tenants = db_all("SELECT id,name,status FROM tenants WHERE status!='DISABLED' ORDER BY id ASC");
  foreach ($tenants as $t) {
    $tid = (int)$t['id'];
    if (invoice_exists($tid, $periodCurrent)) {
      log_line(" - Tenant {$tid}: invoice exists, skip");
      continue;
    }

    // Ensure snapshot exists for this period (should have been created on 25th previous month)
    if (!billing_snapshot_exists($tid, $periodCurrent)) {
      create_billing_snapshot($tid, $periodCurrent);
      // mark note for missing
      db_exec("UPDATE billing_snapshots SET note_json=? WHERE tenant_id=? AND snapshot_for_period_ym=?",
        [json_encode(['rule'=>'snapshot_missing_autogen'], JSON_UNESCAPED_UNICODE), $tid, $periodCurrent]
      );
      log_line(" - Tenant {$tid}: snapshot missing, auto-created");
    }

    $invId = create_invoice_from_snapshot($tid, $periodCurrent, null);
    if (!$invId) {
      log_line(" - Tenant {$tid}: invoice create failed (no snapshot?)");
      continue;
    }

    $inv = db_one("SELECT * FROM invoices WHERE id=?", [$invId]);
    $amt = (int)($inv['total_amount'] ?? 0);
    $due = $inv['due_date'] ?? '';

    notify_create($tid, null, 'INVOICE_CREATED', 'New invoice created', "Invoice #{$invId} created for {$periodCurrent}. Due: {$due}. Amount: {$amt} BDT", "index.php?page=invoice_pay&id={$invId}");
    telegram_send($tid, "🧾 Invoice Created\nTenant: {$t['name']} (#{$tid})\nPeriod: {$periodCurrent}\nInvoice: #{$invId}\nAmount: {$amt} BDT\nDue: {$due}");
    telegram_send(null, "🧾 Tenant Invoice\nTenant #{$tid} ({$t['name']})\nInvoice #{$invId} Period {$periodCurrent} Amount {$amt} BDT Due {$due}");
    audit_log($tid, null, 'INVOICE_CREATED', 'invoice', $invId, ['period'=>$periodCurrent,'amount'=>$amt,'due'=>$due]);

    log_line(" - Tenant {$tid}: invoice #{$invId} created amount={$amt}");
  }
}

// ---------- (C) Daily: suspend tenants with overdue unpaid invoice ----------
log_line("Checking overdue invoices...");
$over = db_all("SELECT i.*, t.status tstatus, t.name tname
                FROM invoices i
                JOIN tenants t ON t.id=i.tenant_id
                WHERE i.status='UNPAID' AND i.due_date < CURDATE() AND t.status='ACTIVE'");

foreach ($over as $inv) {
  $tid = (int)$inv['tenant_id'];
  $iid = (int)$inv['id'];
  db_exec("UPDATE tenants SET status='SUSPENDED_UNPAID', suspended_reason=?, suspended_at=NOW() WHERE id=?",
    ["Unpaid invoice #{$iid} ({$inv['period_ym']})", $tid]
  );

  notify_create($tid, null, 'TENANT_SUSPENDED', 'Billing activities suspended', "Due to unpaid invoice #{$iid} (Period {$inv['period_ym']}), your billing activities are suspended. Please pay now.", "index.php?page=invoice_pay&id={$iid}");
  telegram_send($tid, "⛔ Account Suspended (Unpaid)\nInvoice #{$iid} Period {$inv['period_ym']}\nPay now to re-enable backups.");
  telegram_send(null, "⛔ Tenant Suspended\nTenant #{$tid} ({$inv['tname']})\nUnpaid invoice #{$iid} overdue.");
  audit_log($tid, null, 'TENANT_SUSPENDED_UNPAID', 'tenant', $tid, ['invoice_id'=>$iid,'period'=>$inv['period_ym']]);

  log_line(" - Tenant {$tid}: suspended (invoice #{$iid} overdue)");
}

log_line("Cron daily end.");
